#!/usr/bin/env python
# vim: set fileencoding=latin-1 :
#blockhosts.py

"""Automatic updates to hosts.allow to block IP addresses based on failed
login accesses for ssh/ftp or any such service.

Script to record how many times "sshd" or other service is being attacked,
and when a particular IP address exceeds a configured number of
failed login attempts, that IP address is added to /etc/hosts.allow with
the deny flag to prohibit access.
Script uses /etc/hosts.allow to store (in comments) count
of failed attempts, and date of last attempt for each IP address
By default, hosts.allow is used, but program can be configured to use any
other file, including /etc/hosts.deny, as needed.
IP addresses with expired last attempt dates (configurable)
can be removed, to keep /etc/hosts.allow size manageable.
This script can be run as the optional command in /etc/hosts.allow
itself, so will kick off only when someone connects to a specific service
controlled by tcpwrappers, or use cron to periodically run this script.

TCP_WRAPPERS should be enabled for all services, which allows use of
hosts.allow file.
hosts_options should also have been enabled, which requires compile time
PROCESS_OPTIONS to be turned on. This allows extensions to the
basic hosts.* file line format.  The extensible language supports lines
of this format in /etc/hosts.allow:
    daemon_list : client_list : option : option ...
See the man pages for hosts_options and hosts_access(5) for more
information.


Null Routing and Packet Filtering Blocking
Many services do not use libwrap, so cannot use TCP_WRAPPERS blocking
methods. Those services can be protected by this script, by using
the null routing, or iptables packet filtering to completely block all
network communication from a particular IP address.
Use the --iproute or --iptables options to enable null routing or
packet filtering blocking.
Root permission for the run of blockhosts.py script is needed, since
only root can change routing tables or install iptables rules. This works
fine if using hosts.access/hosts.deny to run this script.
Null routing/packet filtering could be used for example, to scan Apache
web server logs, and based on that, block an IP address so neither
Apache or any other service on the computer will see any network
communication that IP address.


Mail Notification Support
Email notifications can be sent periodically using a cron script, or
email can be sent provided a a given IP address is being blocked by
blockhosts. Such email notifications include all currently blocked
IP addresses in the email message. Will not send email if given IP address
is not yet blocked, or if not a single address is being blocked. SMTP is
required for sending email.

Example hosts.allow script:
Warnings:
* Be sure to keep a backup of your initial hosts.allow (or hosts.deny)
  file, in case it gets overwritten due to an error in this script.
* Do read up on the web topics related to security, denial-of-service,
  and IP-address spoofing.
  Visit the blockhosts home page for references.
* This script handles IPv4 addresses only.

Usage:
For more info, run this program with --help option.

The blockfile (hosts.allow, or if needed, hosts.deny) layout needs to
have a certain format:
  Add following sections, in this order:
  -- permament whitelist and blacklist of IP addresses
  -- blockhosts marker lines - two lines
  -- execute command to kick off blockhosts.py on connects to services

See "man 5 hosts_access" and "man hosts_options" for more details on
hosts.* files line formats.

The two HOSTS_MARKER_LINEs define a section, this is the
region where blockhosts will read/write IP blocking data in the
hosts.allow file. It will use comments to store bookkeeping data needed
by this script in that section, too.
Lines before and after the two HOST_MARKER_LINEs will be left unchanged
in the hosts.allow file

See the "INSTALL" file in the blockhosts.py source package for a
detailed example of the hosts.allow file.

====
Requirements:
    1: Python 2.3 or later, need the optparse module.

    2: Primarily uses host control facility and related files such as
       hosts.access. If not using TCP/IP blocking, then the extensions to
       the access control language as described in the man 5 hosts_options
       page are required, which allow use of :allow and :deny keywords.
       ["...extensions  are  turned  on  at program build time by
       building with -DPROCESS_OPTIONS..."]

    3: If not using host control facilities (tcpd, hosts.access, etc),
       then there needs to be a way to trigger the run of blockhosts.py,
       or blockhosts.py should be run periodically using cron. Secondly,
       there must be some way to update a file to list the blocked ip
       (for example, hosts.deny file, or Apache .htaccess file, etc).
       Alternately, all TCP/IP communication can be blocked by using the
       null-routing or packet filtering options of blockhosts.py
       
====
BlockHosts Script License
This work is hereby released into the Public Domain.
To view a copy of the public domain dedication, visit
http://creativecommons.org/licenses/publicdomain/ or send a letter to
Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Author: Avinash Chopde <avinash@acm.org>
Created: May 2005
http://www.aczoom.com/cms/blockhosts/

"""

# script metadata, also used by setup.py
SCRIPT_ID="blockhosts"
VERSION="2.0.3"
VERSION_DATE="May 2007"
AUTHOR="Avinash Chopde"
AUTHOR_EMAIL="avinash@acm.org"
URL="http://www.aczoom.com/cms/blockhosts/"
LICENSE="http://creativecommons.org/licenses/publicdomain/"
DESCRIPTION="Block IP Addresses based on system logs showing patterns of undesirable accesses."
LONG_DESCRIPTION="""Block IP Addresses based on login or access failure
information in system logs.

Updates a hosts blockfile (such as hosts.allow) automatically,
to block IP addresses. Will also expire previously blocked addresses
based on age of last failed login attempt, this keeps the blockfile
size manageable.
In addition to TCP_WRAPPERS, can also execute iptables or ip route commands
to block all TCP/IP network input from an IP address, so all services, even
those that do not run under libwrap TCP_WRAPPERS, can be protected.

An email notification facility is also available.

"""

import os
import os.path
import sys
import traceback
import time
import errno
import fcntl
import ConfigParser
import syslog
import re
try:
    from optparse import OptionParser, OptionGroup
except ImportError, e:
    print "Missing module: optparse\nWill not work with earlier python versions - 2.3 or later needed.\n", e
    raise

# -------------------------------------------------------------
# This script was inspired by: DenyHosts, which has been developed
#    by Phil Schwartz: http://denyhosts.sourceforge.net/
#
# Mail: 29/12/06 patch by Erik Ljungström      erik [-at-] ibiblio dot 0rg
#    http://www.aczoom.com/cms/forums/blockhosts/patch-enabling-email-alerts
# -------------------------------------------------------------

# ======================= LOGGING FUNCTIONS ========================

def die(msg, *args):
    """Exit, serious error occurred"""

    string = "FATAL ERROR: " + " ".join([str(msg)] + map(str, args))
    print >> sys.stderr, string

    syslog.syslog(syslog.LOG_ERR, string)
    sys.exit(1)

# --------------------------------

class Log:
    """Log support variables and functions, including keeping track
       of last few messages at each level

    """

    # logging levels - each higher level includes lower level messages
    MESSAGE_LEVEL_ERROR = 0    # 0 -> error
    MESSAGE_LEVEL_WARNING = 1  # 1 -> warning
    MESSAGE_LEVEL_INFO = 2     # 2 -> info
    MESSAGE_LEVEL_DEBUG = 3    # 3 -> debug

    # level to use for this run of the program, set in config or command line
    MESSAGE_LEVEL = MESSAGE_LEVEL_WARNING

    # store all messages here, to be sent out in email, if so configured
    MESSAGE_ARCHIVE = []

    def SetPrintLevel(cls, level):
        """Set message level to determine die, error, info, debug print outs.
        
        verbosity_level is the value assigned to options.verbose by the
        OptionParser
        """
        if cls.MESSAGE_LEVEL_ERROR <= level <= cls.MESSAGE_LEVEL_DEBUG:
            cls.MESSAGE_LEVEL = level 
        else:
            raise IndexError, "Invalid Log message level: %s" % str(level)

    SetPrintLevel = classmethod(SetPrintLevel)

    def PrintLevel(cls, level, msg, *args):
        """Print message to stderr, but only if level is >= MESSAGE_LEVEL"""

        string = " ".join([str(msg)] + map(str, args))
        if cls.MESSAGE_LEVEL >= level :
            print >> sys.stderr, string
            # store messages, may be used to send in email notifications
            cls.MESSAGE_ARCHIVE.append(string)
            # keep archive from becoming too large
            if len(cls.MESSAGE_ARCHIVE) > 1024:
                del cls.MESSAGE_ARCHIVE[0]

    PrintLevel = classmethod(PrintLevel)

    def Error(cls, msg, *args):
        """Print error message, a level 0 message, using print_level"""
        cls.PrintLevel(cls.MESSAGE_LEVEL_ERROR, "ERROR: ", msg, *args)
        string = " ".join([str(msg)] + map(str, args))
        syslog.syslog(syslog.LOG_ERR, string)
    Error = classmethod(Error)

    def Warning(cls, msg, *args):
        """Print warning message, a level 1 message, using print_level"""
        cls.PrintLevel(cls.MESSAGE_LEVEL_WARNING, "  Warning: " + msg, *args)
    Warning = classmethod(Warning)

    def Info(cls, msg, *args):
        """Print info message, a level 2 message, using print_level"""
        cls.PrintLevel(cls.MESSAGE_LEVEL_INFO, msg, *args)
    Info = classmethod(Info)

    def Debug(cls, msg, *args):
        """Print debug message, a level 3 message, using print_level"""
        cls.PrintLevel(cls.MESSAGE_LEVEL_DEBUG, msg, *args)
    Debug = classmethod(Debug)

# ======================= CONFIGURATION CLASSES ========================
# defaults for parameters follow this order:
# 1 -> use the value provided as an argument in argv[] to this script
# 2 -> if not, then use the value defined in CONFIGFILE
# 3 -> if not, then use the value hard-coded in this script - HC_OPTIONS

class Config(object):
    """
    Keep track of configuration - priority order: values provided on
    command line, then in the config file then program hard-coded
    defaults.
    """

    HC_OPTIONS = {
        "CONFIGFILE": "/etc/blockhosts.cfg",
        }

    # --------------------------------
    # Class Variables - Start Time Values, Time Formats

    # global time definitions, may be used by other scripts importing blockhosts
    START_TIME = time.time()

    # use ISO time formats to display time, store and decode in /etc/hosts.allow
    ISO_STRFTIME = "%Y-%m-%d %H:%M:%S %Z"
    # %z is better than %Z, but python2.4 has bug - always displays as +0000,
    # so can't use %Z%z which would be preferable for human-readable displays
    # - but note that %z is not accepted by strptime, so stick with %Z for now
    # so, instead of using single time format, need to use another one for UTC
    ISO_UTC_STRFTIME = "%Y-%m-%d %H:%M:%S+0000"

    START_TIME_STR = time.strftime(ISO_STRFTIME, time.localtime(START_TIME))
    START_TIME_UTC_STR = time.strftime(ISO_UTC_STRFTIME, time.gmtime(START_TIME))

    #before version 1.0.4, block file hosts.allow used date/time like this:
    #bh: ip:   200.21.18.136 :   8 : 2007-02-22-14-20
    # to support reading the old format, use the following variables; remove
    # all support for old times after 2008 or so, if everyone has upgraded...
    PRE104_STRFTIME = "%Y-%m-%d-%H-%M"
    PRE104_STRFTIME_RE = re.compile(r"^\d+-\d+-\d+-\d+-\d+$")

    # constants, to recognize markers in the blockfile
    HOSTS_MARKER_LINE       = "#---- BlockHosts Additions"
    HOSTS_MARKER_WATCHED    = "#bh: ip:"
    HOSTS_MARKER_FIRSTLINE  = "#bh: first line:"
    HOSTS_MARKER_OFFSET     = "#bh: offset:"
    HOSTS_MARKER_LOGFILE    = "#bh: logfile:"

    # --------------------------------
    class BHOptionParser(OptionParser):
        def error(self, msg):
            """Print message and exit"""
            # this allows message to get into syslog, so does not get
            # lost if just printed to stdout as base OptionParser does
            die("OptionParser: ", msg)

    # --------------------------------
    def __init__(self, args, ver, desc):

        self._args = args

        self._oparser = Config.BHOptionParser(version=ver, description=desc)

        self._oparser.set_defaults(configfile=self.HC_OPTIONS["CONFIGFILE"])
        self._oparser.add_option("--configfile", type="string", metavar="FILE",
            help="Name of configuration file to read. A configuration file must be readable. (%s)" % self.HC_OPTIONS["CONFIGFILE"])

        # self.config first stores all the values from hard coded program
        # defaults.
        # Its values will be updated from the configuration file values
        defaults = self._oparser.get_default_values()

        self._config = {}
        self._config["CONFIGFILE"] = defaults.configfile

        # data from self.config, updated with values from command-line
        # options - this will sent to parse_args, to use as
        # optparse.Values instance, this is what will be used by the
        # program to read values for all config options

        # check option arguments to see if a config file has been specified
        # note: accepts --configfile=<name>, errors on --configfile <name>
        carg = [arg for arg in args if arg.startswith('--configfile')]

        if carg:
            (self._options, rest_args) = self._oparser.parse_args(carg)
            self._config["CONFIGFILE"] = self._options.configfile

        # print "debug: Config filename: ", self["configfile"]

    def __str__(self):
        return "Configuration: " + str(self._config) + "\nOptions: " + str(self._options)

    def add_section(self, section):
        # add all program hard-coded defaults
        self._config.update(section.HC_OPTIONS)

        # load up the config from the specified config file
        self._load_configfile(section.NAME)

        section.setup_options(self._oparser, self._config)

    def parse_args(self):
        (self._options, rest_args) = self._oparser.parse_args(self._args)
        return rest_args

    def get(self, option):
        """Find value assigned to option in command-line, configfile, or
        hard-coded in program

        Note that case matters, all command line options are lower case,
        and all configuration file options are upper case
        """

        try:
            val = getattr(self._options, option)
            # print "debug: got optparse ", option, ", val ", val
        except AttributeError:
            val = self._config[option]
            # print "debug: failed optparse ", option, ", got config val ", val

        return val

    def __getitem__(self, option):
        return self.get(option)

    # --------------------------------
    def _load_configfile(self, section):
        """Read in the configuration file, given section."""

        filedata = ConfigParser.SafeConfigParser()
        filedata.optionxform = str # leaves tags same case - upper/lower

        configfile = self._config["CONFIGFILE"]

        if not os.path.isfile(configfile):
            # not an error, skip over reading the config file
            die("Could not open config file (%s), cannot continue." % configfile)
            return

        try:
            filedata.read(configfile)
        except Exception, e:
            traceback.print_exc()
            die("Failed reading config file:", e)

        #debug print " loading config %s section %s" % (configfile, section)

        try:
            allitems = dict(filedata.items(section))
        except ConfigParser.NoSectionError:
            traceback.print_exc()
            die("Config file (%s) missing required section (%s)" % (configfile, section))

        keys = allitems.keys()
        for key in keys:
            if key in self._config:
                try:
                    self._config[key] = eval(allitems[key])
                    #debug print " got config %s = %s" % (key, self._config[key])
                except Exception, e:
                    die("Config file Error: invalid line or value found for (%s):\n%s\n" % (key, allitems[key]), e)
            else:
                die("Config file Error: found invalid/unneeded definition:", key)


# --------------------------------
class ConfigSection(object):
    """Abstract base class - all following members need to be defined."""
    NAME = "Undefined"
    HC_OPTIONS = {}
    def setup_options(self, option_parser, config_dict):
        raise NotImplemented

class CommonConfig(ConfigSection):
    """
    Keep track of common configuration, command line options, used by all
    utilities - blockfile reader/updater, mail notifications, ip route
    blocking.
    """

    # Defaults, hard-coded options, these values are used last if no args
    # and no config file provided
    HC_OPTIONS = {
        "VERBOSE": Log.MESSAGE_LEVEL,
        "HOSTS_BLOCKFILE": "/etc/hosts.allow",
        "HOST_BLOCKLINE": ("ALL: ", " : deny",),
            # the line to output, with Host Ip Address in between the
            # strings above, to turn on blocking of that IP address
    }

    NAME = "common"  # config file section name is [NAME]

    def setup_options(self, oparser, config):
        """Update the parser with values for defaults and option parsing
        """
        oparser.set_defaults(verbose=config["VERBOSE"],
            dry_run=False,
            echo="",
            blockfile=config["HOSTS_BLOCKFILE"],
            blockline=config["HOST_BLOCKLINE"],
            )
        defaults = oparser.get_default_values()

        oconfig = OptionGroup(oparser, "Common options",
        """Each option is shown below with its current value in parentheses ().
Nearly all of these options can be specified in the configuration file,
and that is the recommended way.
""")

        oconfig.add_option("-q", "--quiet",
            action="store_const", const=0, dest="verbose",
            help="Be as quiet as possible - only print out error messages")

        oconfig.add_option("-v", "--verbose",
            action="store_const", const=2, dest="verbose",
            help="Be verbose - print errors, warnings, and info messages")

        oconfig.add_option("-g", "--debug",
            action="store_const", const=3, dest="verbose",
            help="Be chatty - print out debug level messages also")

        oconfig.add_option("--dry-run", action="store_true",
            help="Don't write the block file or send email or block routes, just print out blockhosts section of output block file file to stdout instead (%s)" % defaults.dry_run)

        oconfig.add_option("--echo", type="string", metavar="TAG",
            help="Prints TAG on stderr and syslog, may be used to identify a run of this script (%s)" % defaults.echo)

        oconfig.add_option("--blockfile", type="string", metavar="FILE",
            help="[Deprecated: use the config file to specify this instead]\nName of hosts-block-file to read/write (%s)" % defaults.blockfile)

        oparser.add_option_group(oconfig)

# ======================= MAIL SECTION ========================
class MailConfig(ConfigSection):
    """Manage setup related to sending of email

    Keep track of configuration, command line options, and general setup.
    """

    # Defaults, hard-coded options, these values are used last if no args
    # and no values in config file
    HC_OPTIONS = {
        "MAIL": False,
        "MAIL_LOG_MESSAGES": True,
        "NOTIFY_ADDRESS": 'root@localhost.localdomain',
        "SMTP_SERVER": "localhost",
        "SMTP_USER": '',
        "SMTP_PASSWD": '',
        "SENDER_ADDRESS": 'BlockHosts <blockhosts-do-not-reply@localhost.localdomain>',
    }

    NAME = "mail"  # config file section name is [NAME]

    def setup_options(self, oparser, config):
        """Update the parser with values for defaults and option parsing

           Calls add_option for all the options used by mail process
        """

        oparser.set_defaults(
            notify_address=config["NOTIFY_ADDRESS"],
            mail=config["MAIL"],
            check_ip="",
            )

        defaults = oparser.get_default_values()

        oconfig = OptionGroup(oparser, "Mail specific options",
            """These options apply to the process of sending email.
Additional configuration options in the config file: MAIL_LOG_MESSAGES (Whether to include log messages in emails).
    """)

        oconfig.add_option("--mail", action="store_true",
            help="Enable e-mail capability - send message with list of blocked and hosts, if any. See --check-ip option also. Additionally, by default, error messages will also be mailed out (%s)" % defaults.dry_run)

        oconfig.add_option("--check-ip", type="string", metavar="IPADDRESS",
            help="Instead of mailing entire list of blocked address, just send email if given IP address is being blocked (%s)" % defaults.check_ip)

        oconfig.add_option("--notify-address", metavar="ADDRESS",
            help="Address to send notification emails to (%s)" % defaults.notify_address)

        oparser.add_option_group(oconfig)

def do_mail(config, blocked_hosts, watched_hosts):
    """send email with list of blocked and/or watched addresses"""

    import smtplib

    # trim the check-ip argument, same regex as used to match log lines
    check_host = None
    if config["check_ip"]:
        regex = re.compile(r"""(::ffff:)?(?P<host>\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})[^\d]*""")
        m = regex.search(config["check_ip"])
        if m:
            try:
                check_host = m.group("host")
            except IndexError:
                die("** Input error: check-ip address not an IP address (%s)?\nPattern: (%s)\n" % (config["check_ip"], regex))

    found_check_host = False

    subject = SCRIPT_ID + ": currently blocked addresses."
    lines = []

    # always include all blocked hosts in output, only send email if
    # there are some blocked hosts
    # split the watched_hosts into two dicts - one containing all blocked
    # hosts with the data, and second containing all watched but not yet
    # blocked addresses
    if blocked_hosts:
        lines.append("Blocked hosts:")
        hosts = sort_by_value(blocked_hosts, reverse = True)
        for host in hosts:
            data = blocked_hosts[host]
            line = " %15s   count: %4d   updated at:  " % (host, data.count)
            t = time.localtime(data.time)
            line += time.strftime(Config.ISO_STRFTIME, t)
            lines.append(line)
            if host == check_host:
                found_check_host = True
                subject = "%s: Blocked  %s" % (SCRIPT_ID, host)
                Log.Info(" ... mail: found check-ip -- ", subject)

        lines.append("")
        # watched_hosts only contains non-blocked hosts, output those also.
        # get list of hosts, sorted by date/time, in descending order
        hosts = sort_by_value(watched_hosts, reverse = True)
        if hosts:
            lines.append("Watched hosts (not yet blocked):")
        for host in hosts:
            data = watched_hosts[host]
            line = " %15s   count: %4d   updated at:  " % (host, data.count)
            t = time.localtime(data.time)
            line += time.strftime(Config.ISO_STRFTIME, t)
            lines.append(line)
        lines.append("")

    # add all the log messages - errors/warnings always added
    # check if any error or warning exists in lines
    found_warnings = False
    test = re.compile(r"(error|warning)", re.IGNORECASE)
    for l in Log.MESSAGE_ARCHIVE:
        if test.search(l):
            found_warnings = True
            break

    # add the log messages
    if found_warnings or (config["MAIL_LOG_MESSAGES"] and Log.MESSAGE_ARCHIVE):
        lines.append("Log messages:")
        lines += Log.MESSAGE_ARCHIVE

    # all done with mail body, now send it
    if config["check_ip"] and (not found_check_host):
        Log.Info(" ... no email, blocked list does not contain check-ip: ", check_host)
    elif blocked_hosts or found_warnings:
        # only send email if check-ip was found in the blocked list or if
        # there was no check-ip argument and there are blocked hosts, or
        # if there are warnings or errors in the log messages
        Log.Info(" ... sending email notification")
        mailer = MailMessage(config, subject, lines)
        try:
            mailer.send_mail(config["dry_run"])
        except smtplib.SMTPException, e:
            Log.Error(e)
    else:
        Log.Info(" ... no email to send.")

# --------------------------------
class MailMessage:
    """Compose an email message, and then send it
    
    Constructor takes an dict with all mail header info, as well as a
    string specifying subject, and an array of strings specifying body of
    message
    """

    def __init__(self, config, subject, lines):
        # mail header info is in the config object
        self.__address = config["notify_address"].replace('\@', '@')
        self.__sender_address = config["SENDER_ADDRESS"].replace('\@', '@')
        self.__smtp_server = config["SMTP_SERVER"]
        self.__smtp_user = config["SMTP_USER"].replace('\@', '@')
        self.__smtp_passwd = config["SMTP_PASSWD"]
        # If smtp_user and passwd is empty, no authentication is necessary

        # mail subject (string) and lines (list of strings)
        self.__subject = subject
        self.__lines = lines

    def send_mail(self, dry_run = False):

        import smtplib

        if len(self.__address) == 0:
            Log.Debug("   ... no email address specified, not sending any mail")
            return

        session = smtplib.SMTP(self.__smtp_server)
        message = "To: " + self.__address
        message += "\nFrom: "+ self.__sender_address
        message += "\nSubject: " + self.__subject + "\n\n"
        message += "\n".join(self.__lines)
        if dry_run:
            print "\n-----", SCRIPT_ID, ": dry-run, email message-------\n"
            print message
            print "-----"
            return

        if len(self.__smtp_user) > 0:
            Log.Debug("%s: calling SMTP login..." % SCRIPT_ID)
            session.login(self.__smtp_user, self.__smtp_passwd)
        Log.Debug("%s: calling SMTP sendmail..." % SCRIPT_ID)
        smtpresult = session.sendmail(self.__sender_address, self.__address, message)
        if smtpresult:
            errstr = ""
            for recip in smtpresult.keys():
                errstr = """Unable to deliver mail to: %s Server responded: %s %s %s"""\
                         % (recip, smtpresult[recip][0], smtpresult[recip][1], errstr)
                raise smtplib.SMTPException, errstr
        
# ======================= TCP/IP BLOCKING SECTION ========================
class IPBlockConfig(ConfigSection):
    """Manage setup related to using ip/iptables commands to block IP addresses

    Keep track of configuration, command line options, and general setup.
    """

    # Defaults, hard-coded options, these values are used last if no args
    # and no values in config file
    HC_OPTIONS = {
        "IPBLOCK": "",
    }

    NAME = "ipblock"  # config file section name is [NAME]

    def setup_options(self, oparser, config):
        """Update the parser with values for defaults and option parsing

           Calls add_option for all the options used by mail process
        """

        oparser.set_defaults(
            ipblock=config["IPBLOCK"],
            )

        defaults = oparser.get_default_values()

        oconfig = OptionGroup(oparser, "TCP/IP level blocking options",
            """These options apply to the process of using ip route/iptables commands to block IP addresses.
Root permission for the run of this script is needed, since
only root can change routing tables or install iptables rules. [This works
fine if using hosts.access/hosts.deny to run this script.]
All communication to the IP address is blocked at route or packet,
therefore, this method of disabling a host will protect even
non-tcpwrapper services.
""")

        oconfig.add_option("--iproute",
            action="store_const", const="iproute", dest="ipblock",
            help="Enable IP address block capability using ip route commands. Using this, all communication to the IP address is blocked at routing table level (%s)" % defaults.ipblock)

        oconfig.add_option("--iptables",
            action="store_const", const="iptables", dest="ipblock",
            help="Enable IP address block capability, using iptables filtering. Using this, all communication to the IP address is blocked at packet filtering level (%s)" % defaults.ipblock)

        oparser.add_option_group(oconfig)

# --------------------------------
def do_ipblock(config, blocked_hosts):
    """Use ip (null-route) or iptables (packet filtering) to block addresses"""

    if config["ipblock"] == "iproute":
        _do_iproute(config, blocked_hosts)
    elif config["ipblock"] == "iptables":
        _do_iptables(config, blocked_hosts)
    else:
        Log.Error("Invalid value for ipblock option (%s), ignoring." % config["ipblock"])

def _do_cmd(cmd, dry_run, expect=None):
    """Executes command, and returns a tuple that is command return
    status if os.WIFEXITED(waitstatus) is true, otherwise returns
    waitstatus as received from commands.getstatusoutput()
    Prints error if expect code is not same as waitstatus
    """

    import commands

    Log.Debug("Running: ", cmd)
    if dry_run:
        return (0, '')

    (waitstatus, output) = commands.getstatusoutput(cmd)
    Log.Debug("   returned waitstatus: ", waitstatus)
    if output.strip():
        Log.Debug("   output: ", output)

    if os.WIFEXITED(waitstatus):
        waitstatus = os.WEXITSTATUS(waitstatus)

    if None != expect != waitstatus:
        Log.Error("Failed command: %s (%d)\n%s" % (cmd, waitstatus, output))

    return (waitstatus, output)

# --------------------------------
def _do_iptables(config, blocked_hosts):

    chain = SCRIPT_ID

    # use a user-defined iptables chain, named as SCRIPT_ID ("blockhosts")
    # a rule is added to the INPUT chain to jump to the blockhosts chain
    # the blockhosts chain uses DROP action for each blocked IP address
    # IP addresses in blockhosts chain will be synced up to the blocked
    # list, so deletions as well as additions may occur

    # to remove the chains created by this program, run these commands
    # as root, "blockhosts" is the chain name (SCRIPT_ID)
    #   iptables --flush blockhosts
    #   iptables --delete INPUT -j blockhosts
    #   iptables --delete-chain blockhosts
    # to see rules:
    #   iptables --list INPUT --numeric
    #   iptables --list blockhosts --numeric

    dry_run = config["dry_run"]
    if dry_run:
        print "Commands (tentative) to run for IPTables filtering:"

    # check that user-defined chain exists
    # iptables --new blockhosts [ok to run multiple times]
    cmd = "iptables --new %s" % chain
    (waitstatus, output) = _do_cmd(cmd, dry_run, None)
    if waitstatus != 0:
        # iptables: Chain already exists
        Log.Debug(" ... user-defined chain %s already exists, or error occurred " % chain)
    else:
        Log.Info(" ... created user-defined chain %s" % chain)

    # check if INPUT chain jumps to user-defined chain
    #   iptables --list INPUT --numeric
    # Outputs: blockhosts  all  --  0.0.0.0/0            0.0.0.0/0
    cmd = "iptables --list INPUT --numeric"
    (waitstatus, output) = _do_cmd(cmd, dry_run, 0)
    if waitstatus != 0:
        return

    drop_regex = r"""%s.+?0.0.0.0""" % chain
    Log.Debug("   pattern to search for INPUT chain jump: ", drop_regex)
    drop_regex = re.compile(drop_regex)
    found = False
    for line in output.splitlines():
        found = found or drop_regex.search(line)
    if not found:
        Log.Info(" ... creating jump from INPUT to %s chain" % chain)
        cmd = "iptables --append INPUT -j %s" % chain
        (waitstatus, output) = _do_cmd(cmd, dry_run, 0)
        if waitstatus != 0:
            return
    else:
        Log.Debug("   jump rule from INPUT to %s chain exists" % chain)

    # get current list of filtered hosts, and do two things:
    # 1 -> delete rule for host, if not on blocked list
    # 2 -> delete host from blocked list, if rule already exists
    # iptables --list blockhosts --numeric
    # DROP       all  --  10.99.99.99          0.0.0.0/0
    cmd = "iptables --list %s --numeric" % chain
    (waitstatus, output) = _do_cmd(cmd, dry_run, 0)
    if waitstatus != 0:
        return

    drop_regex = r"""DROP.+?(::ffff:)?(?P<host>\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})"""
    Log.Debug("   pattern to search for iptables blocked ip: ", drop_regex)
    drop_regex = re.compile(drop_regex)

    blocked = blocked_hosts.copy()
    for line in output.splitlines():
        m = drop_regex.search(line)
        if not m: continue
        try:
            host = m.group("host")
            if host in blocked:
                del blocked[host]
                Log.Debug("  rule already exists for host ", host)
            else:
                Log.Debug("  rule found for non-blocked host, removing from chain ", host)
                cmd = "iptables --delete %s --source %s -j DROP" % (chain, host)
                Log.Info(" ... iptables, removing rule to block: ", host)
                (waitstatus, output) = _do_cmd(cmd, dry_run, 0)
                if waitstatus != 0:
                    return
        except IndexError:
            pass

    # now blocked contains all IP addresses that need to have DROP rules
    for host in blocked:
        cmd = "iptables --append %s --source %s -j DROP" % (chain, host)
        Log.Info(" ... iptables, adding rule to block: ", host)
        (waitstatus, output) = _do_cmd(cmd, dry_run, 0)
        if waitstatus != 0:
            return

# --------------------------------
def _do_iproute(config, blocked_hosts):
    """Use ip route routing table to block addresses.

    Will delete IP addresses from route if they are no longer blocked,
    and only add new IP addresses if they are not yet being blocked.

    """

    # http://www.tummy.com/journals/entries/jafo_20060727_140652

    dry_run = config["dry_run"]
    if dry_run:
        print "Commands (tentative) to run for ip null-route blocking:"

    # get current list of blackhole'd hosts, and do two things:
    # 1 -> delete route for host, if not on blocked_hosts
    # 2 -> delete host from blocked_hosts, if route already exists
    # ip route list [table <id>]
    # 10.99.99.98 via 127.0.0.1 dev lo

    via = "127.0.0.1"
    cmd = "ip route list"
    (waitstatus, output) = _do_cmd(cmd, dry_run, 0)
    if waitstatus != 0:
        return

    drop_regex = r"""(::ffff:)?(?P<host>\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}).+?via\s+%s""" % via
    Log.Debug("   pattern to search for ip route blocked ip: ", drop_regex)
    drop_regex = re.compile(drop_regex)

    blocked = blocked_hosts.copy()
    for line in output.splitlines():
        m = drop_regex.search(line)
        if not m: continue
        try:
            host = m.group("host")
            if host in blocked:
                del blocked[host]
                Log.Debug("  route already exists for host ", host)
            else:
                Log.Debug("  route found for non-blocked host, removing ", host)
                cmd="ip route del %s" % host
                (waitstatus, output) = _do_cmd(cmd, dry_run, 0)
                if waitstatus != 0:
                    return
                Log.Info(" ... ip route, removing null routing for: ", host)
        except IndexError:
            pass

    # now blocked contains all IP addresses that need to have null-routes
    for host in blocked:
        Log.Info(" ... ip route, adding null route for: ", host)
        cmd = "ip route add %s via %s" % (host, via)
        (waitstatus, output) = _do_cmd(cmd, dry_run, 0)
        if waitstatus != 0:
            return

# ======================= HELPER CLASSES ========================
def sort_by_value(d, reverse = False):
    """ Returns the keys of dictionary d sorted by their values """
    items=d.items()
    backitems=[ [v[1],v[0]] for v in items]
    backitems.sort()
    if reverse:
        backitems.reverse()
    return [ backitems[i][1] for i in range(0,len(backitems))]

class HostData:
    """
    simple record structure to keep track of count seen and time last seen
    for a particular IP host address

    .count is in integer
    .time is same as time.time() - secs since the epoch (1970)
    """
    def __init__(self, count=0, secs = None):
        self.count = count
        self.time = secs

    def __repr__(self):
        return "HostData(" + repr(self.count) + ", " + repr(self.time)  + ")"

    def __cmp__(self, other):
        return cmp(self.time, other.time)

# ======================= EXCEPTIONS ========================
class Error(Exception):
    """Base class for exceptions in this module."""
    pass

class MissingMarkerError(Error):
    "Error: No blockhosts marker found in blockfile (hosts.*) file."
    pass

class SecondMarkerError(Error):
    "Error: Blockhosts section in blockfile (hosts.*) missing second marker."
    pass

# ======================= BLOCKHOSTS SECTION ========================
# Classes: BlockHostsConfig, LockFile, SystemLog, SystemLogOffset, BlockHosts

class BlockHostsConfig(ConfigSection):
    """Manage setup related to handling a blockfile (hosts.allow)

    Keep track of configuration, command line options, and general setup.
    """

    # Defaults, hard-coded options, these values are used last if no args
    # and no config file provided
    HC_OPTIONS = {

        "LOGFILES": ("/var/log/secure",),
            # default list of logs to process, multiple files can be listed

        "COUNT_THRESHOLD": 7,
            # number of invalid attempts after which host is blocked
            # note that actual denial make take one or more attempts - depends
            # on the timing of when LOGFILES are updated by the system,
            # and when this script gets to run

        "AGE_THRESHOLD": 12,
            # number of hours after which host entry is discarded from
            # hosts.allow 24 -> one day, 168 -> one week, 720 -> 30 days,
            # integer values only most attackers go away after they
            # are blocked, so to keep hosts.allow file size small,
            # no reason to make this any more than, say, half-a-day

        "LOCKFILE": "/tmp/blockhosts.lock",
            # need create/write access to this file, used to make sure
            # only one instance of blockhosts.py script writes the
            # HOSTS_BLOCKFILE at one time 
            # note that the mail/iptables/iproute parts of the program
            # do not serialize

        "LOAD_ONLY": False,
            # don't update blockfile, just read it, and prepare list of
            # blocked and watched hosts, possibly for emailing it out, or
            # to update ip/iptables blocks

        ##############################################################
        # ALL_REGEXS_STR: All expressions that match a failed entry.
        # Compulsory: P<host> group, matching the IP address - n.n.n.n - is
        # used to determine IP address to block in each pattern below
        # P<pid> is optional below - use only if expecting multiple regexs
        # to match a failure, but should be counted as one failure only, if
        # the PID in all the matched log lines is the same. SSHD expressions
        # have P<pid> but vsftpd doesn't. Vsftpd keeps the same process
        # active for any number of login failures on that connection,
        # so the vsftpd regex does not include the P<pid> group.
        # 2007: the list of regexs is getting large, so moving all these
        # out to the blockhosts.cfg file, from now, use that file to
        # add/remve regexs

        "ALL_REGEXS": {}, # re.compile'd versions stored here

        "ALL_REGEXS_STR": {}, # blockhosts.cfg file has all the patterns
        }

    NAME = "blockhosts"  # config file section name is [NAME]

    def setup_options(self, oparser, config):
        """Update the parser with values for defaults and option parsing

           Calls add_option for all the options used by blockhosts
        """

        oparser.set_defaults(
            load_only=False,
            logfiles=",".join(config["LOGFILES"]),
            ignore_offset=False,
            blockcount=config["COUNT_THRESHOLD"],
            discard=config["AGE_THRESHOLD"],
            lockfile=config["LOCKFILE"],
            )

        defaults = oparser.get_default_values()

        oconfig = OptionGroup(oparser, "BlockHosts blockfile specific options",
        """These options apply to the process of updating the list of
blocked hosts in the blockfile.
Note that many of these options are deprecated, use the config file to
specify them instead of using the command-line.
""")

        oconfig.add_option("--load-only",
            action="store_true",
            help="Load the blockfile, the blocked/watched host list, but do not prune/add or write back the data (%s)" % defaults.load_only)

        oconfig.add_option("--ignore-offset",
            action="store_true",
            help="Ignore last-processed offset, start processing from beginning.  This is useful for testing or special uses only. (%s)" % defaults.ignore_offset)

        # logfiles are handled specially - since optparse can't do
        # eval(), and I did not want to add a new optparse type, command
        # line arg for logfiles only accepts string, unlike the config file,
        # which accepts the full python syntax - list elements, characters
        # escaped as needed, etc.  Therefore, command line is one string
        # separated by ",", while config file is a python list with multiple
        # filenames -- command line use is deprecated, use configuration
        # file only to specify list of log file names
        oconfig.add_option("--logfiles", type="string", metavar="FILE1,FILE2,...",
            help="[Deprecated: use the config file to specify this instead] The names of log files to parse (\"%s\")" % defaults.logfiles)

        oconfig.add_option("--blockcount", metavar="COUNT", type="int",
            help="[Deprecated: use the config file to specify this instead] Number of invalid tries allowed, before blocking host (%d).  Integer values only." % defaults.blockcount)

        oconfig.add_option("--discard", type="int", metavar="AGE",
            help="[Deprecated: use the config file to specify this instead] Number of hours after which to discard record - if most recent invalid attempt from IP address is older, discard that host entry (%d).  Integer values only." % defaults.discard)

        oconfig.add_option("--lockfile", metavar="FILE",
            help="[Deprecated: use the config file to specify this instead] Prevent multiple instances from writing to blockfile at once - open this file for locking and writing (%s)" % defaults.lockfile)

        oparser.add_option_group(oconfig)

        # compile all regular expression patterns
        for (name, pattern) in config["ALL_REGEXS_STR"].iteritems():
            config["ALL_REGEXS"][name] = re.compile(pattern)
            


class LockFile:
    """Create exclusive advisory lock on given file, which must be opened
    for write access atleast
    """
    def __init__(self, path):
        self._path = path
        self._locked = 0

    def lock(self):
        try:
            self._fp = open(self._path, "a+") # a+ prevents trashing the file!
        except IOError, e :
            if e.errno == errno.ENOENT: # no such file
                # "w+" will trash existing file, or create new one
                self._fp = open(self._path, "w+")
                Log.Debug(" ... first r+ lock file open failed, so opened with w+ mode")
            else:
                raise

        try:
            rv = fcntl.lockf(self._fp.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB)
        except IOError, e :
            if e.errno == errno.EAGAIN:
                Log.Debug("File (%s) already locked, EAGAIN." % self._path)
            elif e.errno == errno.EACCES:
                Log.Debug("File (%s) permission denied, EACCES." % self._path)
            else:
                Log.Debug("File (%s) fcntl.lockf failed." % self._path, e)
            raise
        else:
            self._locked = 1


    def unlock(self):
        if not self._locked:
            Log.Debug("  debug warning: LockFile: called unlock when no lock was held, file ", self._path)
            return

        try:
            rv = fcntl.lockf(self._fp.fileno(), fcntl.LOCK_UN)
            self._fp.close()
        except IOError, e:
            Log.Debug("  debug warning: LockFile: failed to unlock or close file ", self._path, e)
        else:
            self._fp = None
            self._locked = 0


    def get_path(self):
        return self._path

# --------------------------------
class SystemLogOffset:
    """Simple record structure to keep track of location into a system
    log like message/secure file.

    Uses a offset, along with the entire first line of the file at the
    time, to allow detection of log rotation
    """
    def __init__(self, offset=0L, first_line=""):
        self.offset = long(offset)
        self.first_line = first_line

    def load_string(self, line):
        if line.startswith(Config.HOSTS_MARKER_OFFSET):
            value = line[ len(Config.HOSTS_MARKER_OFFSET) : ]
            try:
                self.offset = long(value.strip())
            except ValueError, e:
                Log.Warning("could not decode offset, using 0:", e)
                self._last_offset = 0
                return False
        elif line.startswith(Config.HOSTS_MARKER_FIRSTLINE):
            value = line[ len(Config.HOSTS_MARKER_FIRSTLINE) : ]
            self.first_line = value.rstrip()
        return True

    def dump_string(self):
        return "%s %ld\n%s%s\n\n" % (Config.HOSTS_MARKER_OFFSET, self.offset,
                                Config.HOSTS_MARKER_FIRSTLINE, self.first_line)

    def __repr__(self):
        return 'SystemLogOffset(%ld, %s)' % (self.offset, repr(self.first_line))

# --------------------------------
class SystemLog:
    """
    Handles read operations on the system log like messages/secure log
    which contains all the sshd/proftpd or other logging attempts.
    Read operations skip previously scanned portion of the log file, if
    that is applicable.
    """
    def __init__(self, logfile):
        self._offset = SystemLogOffset()
        self._logfile = logfile
        self._fp = None

    def open(self, offset):
        try:
            self._fp = open(self._logfile, "r")
            self._offset.first_line = self._fp.readline()[:-1]
            self._fp.seek(0, 2)
            self._offset.offset = self._fp.tell()
        except IOError:
            traceback.print_exc()
            die("Can't open or read: %s" % self._logfile)
            sys.exit(1)

        Log.Debug("SystemLog open:")
        Log.Debug("   first_line:", self._offset.first_line)
        Log.Debug("   file length:", self._offset.offset)

        if self._offset.first_line != offset.first_line:
            # log file was rotated, start from beginning
            self._offset.offset = 0L
            Log.Debug("   log file rotated, ignore old offset, start at 0")
        elif self._offset.offset > offset.offset:
            # new lines exist in log file, start from old offset
            self._offset.offset = offset.offset
        else:
            # no new entries in log file
            # Log.Debug("   log file offset unchanged, nothing new to read")
            pass

        Log.Info(" ... loading log file, offset:", self._logfile, self._offset.offset)

        self._fp.seek(self._offset.offset)

        return self._fp != None

    def close(self):
        try:
            return self._fp.close()
        except IOError, e:
            Log.Warning("could not close logfile ", self._logfile, e)

        return None

    def readline(self):
        try:
            line = self._fp.readline()
            self._offset.offset = self._fp.tell()
        except IOError, e:
            line = None
            Log.Warning("readline: could not read logfile", self._logfile, e)

        return line

    def get_offset(self):
        return self._offset


# --------------------------------
class BlockHosts:
    def __init__(self, blockfile, host_blockline):
        self._watched_hosts = {} # hosts -> HostData [count, last seen]
        self._blocked_ips = [] # ip addressess blocked
        self._offset_first_marker = -1L
        self._remaining_lines = [] # all lines after the 2nd end marker
        self._blockfile = blockfile
        self._host_blockline = host_blockline

        # pattern to get IP address from a blocked IP address line
        # in between the blockhosts marker section in blockfile
        self._blocked_regex = r"""\s*(::ffff:)?(?P<host>\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})\s*""".join(host_blockline)
        Log.Debug("   pattern to search for blocked ip: ", self._blocked_regex)
        self._blocked_regex = re.compile(self._blocked_regex)


    def load_hosts_blockfile(self, logoffsets = {}):
        self._remaining_lines = []

        Log.Info(" ... load blockfile:", self._blockfile)

        state = 0
        # state = 0 -> error state
        # state = 1 -> have not seen first marker
        # state = 2 -> have seen first marker, not seen second marker
        # state = 3 -> have seen second marker
        found_first_marker = False
        try:
            fp = open(self._blockfile, "r")
            state = 1
            # skip all lines to first marker
            while fp and state < 2:
                offset = fp.tell()
                line = fp.readline()
                if not line: break

                line = line.strip()
                if not line: continue

                # Log.Debug("1: got line: ", line)
                if line.startswith(Config.HOSTS_MARKER_LINE):
                    self._offset_first_marker = offset
                    found_first_marker = True
                    Log.Debug(" ... seen all state 1 lines, now inside blockhosts markers at offset ", offset)
                    state = 2

            if not found_first_marker:
                raise MissingMarkerError

            # read all lines to second marker, fill in watched_hosts
            state = self._process_state_2(fp, logoffsets, line)

            # read all lines from second marker to end of file
            if fp and state == 3:
                Log.Info(" ... found both markers, count of hosts being watched:", len(self._watched_hosts))
                self._remaining_lines = fp.readlines()

            fp.close()

        except IOError, e:
            Log.Error("could not read block-file, last state: ", state)
            state = 0
            raise

        Log.Debug("block-file: Got initial watched hosts data:")
        Log.Debug(self._watched_hosts )
        Log.Debug("-------------------")
        Log.Debug("block-file: Got remaining lines:")
        Log.Debug(self._remaining_lines)
        Log.Debug("-------------------")

        return state > 2

    # --------------------------------
    def _process_state_2(self, fp, logoffsets, line):
        state = 2
        logfile = ""

        found_second_marker = False
        while fp and state == 2:
            line = fp.readline()
            if not line: break

            line = line.strip()
            if not line: continue

            if line.startswith(Config.HOSTS_MARKER_LINE):
                found_second_marker = True
                state = 3
            elif line.startswith(Config.HOSTS_MARKER_LOGFILE):
                logfile = line[ len(Config.HOSTS_MARKER_LOGFILE) : ]
                logfile = logfile.strip()
                Log.Debug("2: found logfile name line: ", logfile)
                logoffsets[ logfile ] = SystemLogOffset()
            elif line.startswith(Config.HOSTS_MARKER_OFFSET) or line.startswith(Config.HOSTS_MARKER_FIRSTLINE):
                if logfile:
                    logoffsets[logfile].load_string(line)
                else:
                    Log.Warning("... log file name not known, ignoring offset or first_line info: ", line)
            elif line.startswith(Config.HOSTS_MARKER_WATCHED):
                line = line[ len(Config.HOSTS_MARKER_WATCHED) : ]

                name, value = line.split(":", 1)
                if not name: return state

                name = name.strip()
                self._watched_hosts[name] = HostData(0, Config.START_TIME)

                if ":" in value:
                    value, datestr = value.split(":", 1)
                    datestr = datestr.strip()
                    try:
                        self._watched_hosts[name].count = int(value)
                        if not Config.PRE104_STRFTIME_RE.match(datestr):
                            # new date/time format
                            self._watched_hosts[name].time = time.mktime(time.strptime(datestr, Config.ISO_STRFTIME))
                        else:
                            # is old date format, remove in 2008 or later
                            self._watched_hosts[name].time = time.mktime(time.strptime(datestr, Config.PRE104_STRFTIME))
                    except ValueError, e:
                        Log.Error("reading date or count for ip:", e)
                        traceback.print_exc()
                        state = 0

                    Log.Debug("2: got host-count-date ", name, value, self._watched_hosts[name].time)

                else:
                    Log.Warning("2: invalid line, no date, just count", name, value)
                    self._watched_hosts[name].count = int(value)
            else:
                # not a blockhosts line, but in between blockhosts markers
                # this is a blocked host, store its ip address
                m = self._blocked_regex.search(line)
                if m:
                    try:
                        host = m.group("host")
                        self._blocked_ips.append(host)
                        Log.Debug("2: found blocked host: %s" % host)
                    except IndexError:
                        Log.Error("Expected to find group <host> in match: ", line)
                else:
                    Log.Warning("Unrecognized line found between blockhosts markers: ", line)

        if not found_second_marker:
            raise SecondMarkerError

        return state

    # --------------------------------
    def increment_host(self, host):
        try:
            stat = self._watched_hosts[host]
        except KeyError:
            self._watched_hosts[host] = HostData()
            stat = self._watched_hosts[host]
            Log.Debug(" ... In increment host, created host entry ", host)

        stat.count += 1
        stat.time = Config.START_TIME
        # date time is aggresive - exact would be to parse the log line,
        # but that much accuracy is not necessary

    # --------------------------------
    def update_hosts_lists(self, count_threshold, prune_time):
        """Prune watched list based on age, and create blocked list"""
        self._blocked_ips = []
        hosts = sort_by_value(self._watched_hosts, reverse = True)
        for host in hosts:
            # first remove all records that are considered old/expired
            # use <= instead of <, to allow --discard=0 to remove all hosts
            data = self._watched_hosts[host]
            if data.time <= prune_time:
                Log.Info("  will remove expired host: ", host, data)
                del self._watched_hosts[host]
                continue

            # check if number of invalid attempts exceeds threshold
            if data.count > count_threshold:
                self._blocked_ips.append(host)

        return(self._blocked_ips, self._watched_hosts)

    # --------------------------------
    def update_hosts_blockfile(self, logoffsets, load_only = False):

        lines = []

        #Log.Debug(" here are new hosts from get_deny_hosts:", self._blocked_ips)

        # first collect all the lines that will go in the blockhosts
        # section of blockfile - this is stored in lines[]
        status = False

        lines.append("%s\n" % Config.HOSTS_MARKER_LINE) # first marker line

        for host in self._blocked_ips:
            lines.append(host.join(self._host_blockline) + "\n")

        if self._blocked_ips: lines.append("\n")

        Log.Debug("Collecting watched_hosts counts info for block-file")
        hosts = sort_by_value(self._watched_hosts, reverse = True)
        for host in hosts:
            date = time.localtime(self._watched_hosts[host].time)
            date = time.strftime(Config.ISO_STRFTIME, date)
            lines.append("%s %15s : %3d : %s\n" % (Config.HOSTS_MARKER_WATCHED, host, self._watched_hosts[host].count, date))
            # Log.Debug("adding line to blockfile: ", host)

        if len(self._watched_hosts) > 0: lines.append("\n")

        Log.Debug("Collecting log file offset info for block-file")
        files = logoffsets.keys()
        for name in files:
            lines.append("%s %s\n" % (Config.HOSTS_MARKER_LOGFILE, name))
            lines.append(logoffsets[name].dump_string())

        lines.append("%s\n" % Config.HOSTS_MARKER_LINE) # second marker line
        lines = lines + self._remaining_lines;

        Log.Info(" ... updates: counts: hosts to block: %d; hosts being watched: %d" % (len(self._blocked_ips), len(self._watched_hosts)))
        if load_only:
            sys.stdout.writelines(lines)
            return True

        # open file in read/write mode
        try:
            fp = open(self._blockfile, "r+")
            try:
                if self._offset_first_marker > -1:
                    # have seen first marker, go to start of first marker
                    fp.seek(self._offset_first_marker)
                else:
                    # no marker, go to end of existing file
                    # may not come here, depends on if not seeing a marker
                    # was considered an error in the load_hosts_blockfile function,
                    # but if it does come here, then don't overwrite any
                    # existing data
                    fp.seek(0, 2)
                    Log.Debug(" no hosts marker found, positioning for writing at end of (%s)" % self._blockfile)

                fp.writelines(lines)
                fp.truncate()
                status = True

                # this could be the final message, no errors to this point, send
                # this info to the system log
                syslog.syslog(syslog.LOG_INFO, "final counts: blocking %d, watching %d" % (len(self._blocked_ips), len(self._watched_hosts)))

            finally:
                fp.close()
        except IOError, e:
            traceback.print_exc()
            Log.Error("Could not update blockfile ", self._blockfile)

        return status

    # --------------------------------
    def get_hosts_lists(self):
        """Return two dicts of blocked and watched hosts.
        
        First dict is of all hosts being blocked, and second dict
        is of all hosts being watched, both contain host -> HostData,
        with hosts, counters and time. No hosts are common between
        the dicts.
        """

        blocked_only = {}
        watched_only = self._watched_hosts.copy()
        for host in self._blocked_ips:
            try:
                data = watched_only[host]
                blocked_only[host] = data
                del watched_only[host]
            except KeyError:
                Log.Error("%s: found blocked IP (%s), but not in watched list (script error?)." % (SCRIPT_ID, host))

        return (blocked_only, watched_only)

# ======================= MAIN ========================

def main():
    """Collect args, open block-file, search log files, block IP addresses"""

    syslog.openlog(SCRIPT_ID)

    config = Config(sys.argv[1:], VERSION, LONG_DESCRIPTION)

    config.add_section(CommonConfig())
    config.add_section(BlockHostsConfig())
    config.add_section(MailConfig())
    config.add_section(IPBlockConfig())

    rest_args = config.parse_args()

    Log.SetPrintLevel(config["verbose"])

    # --------------------------------
    Log.Info("%s %s started: %s" % (SCRIPT_ID, VERSION, Config.START_TIME_STR))

    Log.Debug("Debug mode enabled.")
    Log.Debug("Got config and options:", config)

    if config["echo"]:
        # force printing echo tag to syslog, and to screen if info enabled
        syslog.syslog(syslog.LOG_INFO, "echo tag: %s" % config["echo"])
        Log.Info(" ... echo tag: %s" % config["echo"])

    if rest_args:
        Log.Warning("ignoring positional arguments - there should be none!", rest_args)

    load_only = config["load_only"] 
    dry_run = config["dry_run"]

    # --------------------------------
    # args check
    if len(config["ALL_REGEXS"]) < 1:
        die("No regular expressions found, maybe missing from (%s)? " %config["CONFIGFILE"])
        
    # --------------------------------
    if not (load_only or dry_run):
        lock = LockFile(config["lockfile"])
        try:
            lock.lock()
        except IOError, e:
            if e.errno == errno.EAGAIN:
                die("Exiting: another instance running? File (%s) already locked" % lock.get_path())
            elif e.errno == errno.EACCES:
                die("Failed to lock: open/write permission denied on (%s)" % lock.get_path())
            else:
                die("Lock error: file (%s), failed to get lock" % lock.get_path(), e)

        Log.Debug("File lock obtained (%s) for excluding other instances" % lock.get_path())

    # --------------------------------
    # load block file data with current list of blocked and watched hosts

    dh = BlockHosts(config["blockfile"], config["blockline"])
    prev_logoffsets = {}
    new_logoffsets = {}
    ip_pid = {}

    try:
        dh.load_hosts_blockfile(prev_logoffsets)
    except (MissingMarkerError, SecondMarkerError):
        die("Failed to load blockfile - block-file marker error\n Expected two marker lines in the file, somewhere in the middle of the file:\n%s\n%s\n" % (Config.HOSTS_MARKER_LINE, Config.HOSTS_MARKER_LINE))
    except:
        traceback.print_exc()
        die("Failed to load blockfile, unexpected error")

    # --------------------------------
    # scan logfiles for IP hosts illegally accessing services, update
    # host IP access failure counters 

    if load_only:
        logfiles = ""
    else:
        logfiles = config["logfiles"].split(",");

    for logfile in logfiles:
        Log.Debug(" ------- looking into log file: ", logfile)
        sl = SystemLog(logfile)

        offset = SystemLogOffset(0,"")
        if not config["ignore_offset"]:
            if logfile in prev_logoffsets:
                offset = prev_logoffsets[logfile]
            else:
                Log.Warning("no offset found, will read from beginning in logfile:", logfile)

        sl.open(offset)

        while 1:
            line = sl.readline()
            if not line: break

            line = line.strip()
            if not line: continue

            regexs = config["ALL_REGEXS"].keys()
            for regex in regexs:
                m = config["ALL_REGEXS"][regex].search(line)
                if m:
                    try:
                        host = m.group("host")
                    except IndexError:
                        die("** Program error: pattern matched line:\n%s\n  but no 'host' group defined in regex: (%s)" % (line, regex))

                    try:
                        pid = m.group("pid")
                        ip_pid_key = host + "-" + pid
                    except IndexError:
                        ip_pid_key = ""

                    # if this hostip and processid already seen before,
                    # then this attempt has already been counted, don't
                    # double count, break out of here.  This may happen
                    # with SSHD-Fail and SSHD-Invalid matches.
                    if ip_pid_key:
                        if ip_pid_key in ip_pid:
                            ip_pid[ip_pid_key] += 1
                            Log.Debug("      ignoring duplicate failure line:", regex, ", IP-pid:", ip_pid_key)
                            break

                    dh.increment_host(host)

                    if ip_pid_key:
                        ip_pid[ip_pid_key] = 1
                        Log.Debug("    found failed access for ", regex, ", IP-pid:", ip_pid_key)
                    else:
                        Log.Debug("    found failed access for ", regex, ", IP:", host)
                    break

        sl.close()

        new_logoffsets[logfile] = sl.get_offset()

        Log.Debug(" ------- finished looking into log file: ", logfile)

    # --------------------------------
    # prune hosts list, determine new blocked and watched lists

    prune_date = Config.START_TIME - config["discard"]*60*60

    Log.Info(" ... will discard all host entries older than ", time.strftime(Config.ISO_STRFTIME, time.localtime(prune_date)))

    if not load_only:
        Log.Debug(" ------- collecting block file updates --- ")
        dh.update_hosts_lists(config["blockcount"], prune_date)
        dh.update_hosts_blockfile(new_logoffsets, dry_run)
        if not dry_run: lock.unlock()

    # ---- send mail
    (blocked, watched) = dh.get_hosts_lists()
    if config["mail"]:
        do_mail(config, blocked, watched)

    # ---- use routing or filtering to block ip addresses
    if config["ipblock"]:
        do_ipblock(config, blocked)

# --------------------------------
if __name__ == '__main__':
    main()
